<?php
/**
 * PayNow_Abstract_Payment_Gateway class file
 *
 * @package paynow
 */

defined( 'ABSPATH' ) || exit;

/**
 * PayNow_Payment main class for handling all checkout related process.
 */
abstract class PayNow_Abstract_Payment_Gateway extends WC_Payment_Gateway {

	/**
	 * Plugin name
	 *
	 * @var string
	 */
	protected $plugin_name;

	/**
	 * Plugin version
	 *
	 * @var string
	 */
	protected $version;

	/**
	 * Web NO
	 *
	 * @var string
	 */
	protected $web_no;

	/**
	 * Trans Password
	 *
	 * @var string
	 */
	protected $trans_pwd;

	/**
	 * Pay Type
	 *
	 * @var string
	 */
	protected $pay_type;

	/**
	 * Merchant Name
	 *
	 * @var string
	 */
	protected $merchant_name;

	/**
	 * Test mode
	 *
	 * @var boolean
	 */
	protected $testmode;

	/**
	 * API url
	 *
	 * @var string
	 */
	protected $api_url;

	/**
	 * Constructor
	 */
	public function __construct() {

		$this->icon              = $this->get_icon();
		$this->has_fields        = false;
		$this->order_button_text = __( 'Proceed to PayNow', 'wc-paynow-payment' );
		$this->supports          = array(
			'products',
		);

		$this->web_no        = strtoupper( get_option( 'paynow_payment_web_no' ) );
		$this->trans_pwd     = get_option( 'paynow_payment_trans_pwd' );
		$this->merchant_name = get_option( 'paynow_payment_merchant_name' ); // 要跟 paynow 會員帳號中的 網站名稱 一樣.
		$this->min_amount    = 30;

		$this->testmode = wc_string_to_bool( get_option( 'paynow_payment_testmode_enabled' ) );
		$this->api_url  = ( $this->testmode ) ? 'https://test.paynow.com.tw/service/etopm.aspx' : 'https://www.paynow.com.tw/service/etopm.aspx';

		add_action( 'woocommerce_order_details_after_order_table', array( $this, 'paynow_payment_detail_after_order_table' ), 10, 1 );
		add_filter( 'woocommerce_thankyou_order_received_text', array( $this, 'paynow_thankyou_order_unpaid_message' ), 10, 2 );

	}

	/**
	 * Payment method settings
	 *
	 * @return void
	 */
	public function admin_options() {
		echo '<h3>' . esc_html( $this->get_method_title() ) . '</h3>';
		echo '<p>' . sprintf( esc_html__( '%1$s is a payment gateway provided by <a href=%2$s target=_blank>PayNow</a>', 'wc-paynow-payment' ), esc_html( $this->get_method_title() ), esc_url( 'https://www.paynow.com.tw' ) ) . '</p>';
		echo '<table class="form-table">';
		$this->generate_settings_html();
		echo '</table>';
	}

	/**
	 * Admin meta boxes
	 *
	 * @deprecated
	 * @return void
	 */
	public function paynow_add_meta_boxes() {
		global $post;
		if ( get_post_meta( $post->ID, '_payment_method', true ) === $this->id ) {
			add_meta_box(
				'paynow-order-meta-boxes',
				__( 'PayNow Payment Detail', 'wc-paynow-payment' ),
				array(
					$this,
					'paynow_admin_meta',
				),
				'shop_order',
				'side',
				'default'
			);
		}
	}

	/**
	 * Display message on order thankyou page.
	 *
	 * @param string   $text Message on thankyou page.
	 * @param WC_Order $order The Order object.
	 * @return string
	 */
	public function paynow_thankyou_order_unpaid_message( $text, $order ) {
		if ( $order ) {
			if ( $order->get_payment_method() !== $this->id ) {
				return $text;
			}

			$tran_status = get_post_meta( $order->get_id(), '_paynow_tran_status', true );
			if ( $order->get_status() == 'pending' || 'F' == $tran_status ) {
				if ( empty( $this->incomplete_payment_message ) ) {
					$text = '<span class="paynow-incomplete-payment-message">' . esc_html__( 'We have received your order, but the payment is incompleted.', 'wc-paynow-payment' ) . '</span>';
				} else {
					$text = '<span class="paynow-incomplete-payment-message">' . $this->incomplete_payment_message . '</span>';
				}
			}
		}

		return $text;
	}

	/**
	 * Check if the gateway is available for use.
	 *
	 * @return bool
	 */
	public function is_available() {
		$is_available = ( 'yes' === $this->enabled );

		if ( WC()->cart && 0 < $this->get_order_total() && 0 < $this->max_amount && $this->max_amount < $this->get_order_total() ) {
			$is_available = false;
		}

		if ( WC()->cart && 0 < $this->get_order_total() && $this->min_amount > $this->get_order_total() ) {
			$is_available = false;
		}

		return $is_available;
	}

	/**
	 * Payment gateway icon output
	 *
	 * @return string
	 */
	public function get_icon() {
		$icon_html  = '';
		$icon_html .= '<img src="' . PAYNOW_PLUGIN_URL . 'paynow-logo.jpg " alt="' . __( 'Paynow Payment Gateway', 'wc-paynow-payment' ) . '" />';
		return apply_filters( 'woocommerce_gateway_icon', $icon_html, $this->id );
	}

	/**
	 * Build pass code for api usage.
	 *
	 * @param string $order_no The order number.
	 * @param string $total_price The order total amount.
	 * @return string
	 */
	public function build_pass_code( $order_no, $total_price ) {
		$pass_code_string = $this->web_no . $order_no . $total_price . $this->trans_pwd;
		return strtoupper( sha1( $pass_code_string ) );
	}

	/**
	 * Return payment gateway method title
	 *
	 * @return string
	 */
	public function get_method_title() {
		return $this->method_title;
	}

	/**
	 * Return PayNow web no
	 *
	 * @return string
	 */
	public function get_web_no() {
		return $this->web_no;
	}

	/**
	 * Return PayNow merchant name
	 *
	 * @return string
	 */
	public function get_merchant_name() {
		return $this->merchant_name;
	}

	/**
	 * Return PayNow payment url
	 *
	 * @return string
	 */
	public function get_api_url() {
		return $this->api_url;
	}

	/**
	 * Return PayNow payment pay type
	 *
	 * @return string
	 */
	public function get_pay_type() {
		return $this->pay_type;
	}

	/**
	 * Build items as string
	 *
	 * @param WC_Order $order The order object.
	 * @return string
	 */
	public function get_items_infos( $order ) {
		$items  = $order->get_items();
		$item_s = '';
		foreach ( $items as $item ) {
			$item_s .= $item['name'] . 'X' . $item['quantity'];
			if ( end( $items )['name'] !== $item['name'] ) {
				$item_s .= ',';
			}
		}
		$resp = ( mb_strlen( $item_s ) > 200 ) ? mb_substr( $item_s, 0, 200 ) : $item_s;
		return $resp;
	}

	/**
	 * Get plugin name
	 *
	 * @return string
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * Get plugin version
	 *
	 * @return string
	 */
	public function get_version() {
		return $this->version;
	}
}
